/*
 *  Copyright (C) 2002-2003 Hongli Lai
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <unistd.h>
#include <prefixdb.h>


static void
usage ()
{
        printf ("Usage: pdbrun ROOTNAME[:ROOTNAME2[:ROOTNAMEn]] COMMAND [ARGS]\n"
                "Read prefix information for ROOTNAME(n), set PATH and LD_LIBRARY_PATH, and run COMMAND [ARGS].\n");
}


static void
version ()
{
        printf ("pdbrun version 0.1\n"
                "Copyright (c) 2002 Hongli Lai\n\n"
                "This is free software; see the source for copying conditions.  There is NO\n"
                "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
}


/* Copied & modified from glib */
static char *
g_strndup (char *str, int n)
{
        char *tmp, *result;

        if (str == NULL)
                return NULL;
        if (n <= 0)
                return NULL;

        tmp = (char *) calloc (sizeof (char), n + 1);
        if (!tmp)
                return NULL;

        result = (char *) memcpy (tmp, str, n);
        if (!result)
            {
                free (tmp);
                return NULL;
            }
        else
                return result;
}


static void
var_append (char **var, char *str)
{
        if (!str)
                return;

        if (!*var)
                *var = strdup (str);
        else
            {
                char *newvar;
                size_t len;

                len = strlen (*var) + strlen (str) + 2;
                newvar = malloc (len + 1);
                snprintf (newvar, len, "%s:%s", str, *var);

                free (*var);
                *var = newvar;
            }
}


/* mode:
   0 = normal
   1 = no mercy
   2 = quiet
 */
static int
run (char *the_rootnames, char **args)
{
        #define safe_strdup(str)  ((str) ? strdup (str) : ((void *) 0))
        char *rootnames = the_rootnames;
        char *path, *ld, *man;

        path = safe_strdup (getenv ("PATH"));
        ld = safe_strdup (getenv ("LD_LIBRARY_PATH"));
        man = safe_strdup (getenv ("MANPATH"));

        /* Split the rootnames */
        while (rootnames != NULL)
            {
                char *found = strstr (rootnames, "::");
                PrefixDB *db;
                char *str;

                if (found)
                    {
                        str = g_strndup (rootnames, found - rootnames);
                        rootnames = (char *) (((int) found) + 2);
                    }
                else
                    {
                        str = strdup (rootnames);
                        rootnames = NULL;
                    }

                /* str is the current rootname */
                db = pdbInit (str, NULL, NULL, NULL, NULL, NULL, NULL);
                if (!db)
                        fprintf (stderr, "pdbrun: rootname %s does not exist\n", str);
                else
                    {
                        char *prefix, *manpath1, *manpath2;

                        var_append (&path, pdbGetBin (db));
                        var_append (&ld, pdbGetLib (db));

                        prefix = pdbGetPrefix (db);
                        if (prefix)
                            {
                                manpath1 = calloc (PATH_MAX + 1, 1);
                                snprintf (manpath1, PATH_MAX, "%s/share/man", prefix);
                                manpath2 = calloc (PATH_MAX + 1, 1);
                                snprintf (manpath2, PATH_MAX, "%s/man", prefix);

                                var_append (&man, manpath1);
                                var_append (&man, manpath2);
                            }

                        pdbFree (db);
                    }
                free (str);
            }

        /* Set the environment variables */
        setenv ("PATH", path, 1);
        setenv ("LD_LIBRARY_PATH", ld, 1);
        setenv ("MANPATH", man, 1);

        /* We're done; execvp() the program */
        execvp (args[0], args);

        /* Error: execvp() failed */
        perror ("pdbrun");
        return 1;
}


static char **
copy_array (char **orig, int argc, int begins)
{
        int count, i, j = 0;
        char **result;

        count = argc - begins;
        result = calloc (count + 1, sizeof (char *));
        for (i = begins; i < argc; i++)
            {
                result[j] = safe_strdup (orig[i]);
                j++;
            }
        return result;
}


int
main (int argc, char *argv[])
{
        if (argc <= 1)
            {
                /* No parameters */
                usage ();
                return 1;
            }
        else if (strcmp (argv[1], "--version") == 0)
            {
                version ();
                return 0;
            }
        else if (strcmp (argv[1], "--help") == 0)
            {
                usage ();
                return 0;
            }
        else if (argc < 3)
            {
                /* Not enough parameters */
                usage ();
                return 1;
            }
        else
            {
                char **args, *rootnames;

                rootnames = argv[1];
                args = copy_array (argv, argc, 2);
                return run (rootnames, args);
            }
}
