/*
 *  Copyright (C) 2002-2003 Hongli Lai
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <prefixdb.h>


static void
usage ()
{
        printf ("Usage: pdbget ROOTNAME OPTION [FALLBACK]\n"
                "Read an entry from libprefixdb's database.\n"
                "\n"
                "Arguments:\n"
                "  --help      Display this help and exit.\n"
                "  --version   Output version information and exit.\n\n"
                "OPTION is one of the following:\n"
                "  --prefix    Get the prefix.\n"
                "  --standard  FALLBACKPREFIX ENTRYNAME\n"
                "  --custom    ENTRYNAME\n"
                "\n"
                "For --standard, ENTRYNAME is one of the following:\n"
                "bin, share, etc, lib, libexec\n");
}


static void
version ()
{
        printf ("pdbget version 0.9.1\n"
                "Copyright (c) 2002 Hongli Lai\n\n"
                "This is free software; see the source for copying conditions.  There is NO\n"
                "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
}


void
printstd (char *name, char *dir, char *fallback)
{
        if (!dir)
            {
                if (fallback)
                        printf ("%s\n", fallback);
                else
                    {
                        fprintf (stderr, "pdbget: information about '%s' not found\n", name);
                        exit (1);
                    }
            }
        else
                printf ("%s\n", dir);
}


int
main (int argc, char *argv[])
{
        char *domain;

        if (argc < 2)
            {
                usage ();
                return 1;
            }
        if (strcasecmp (argv[1], "--help") == 0)
            {
                usage ();
                return 0;
            }
        else if (strcasecmp (argv[1], "--version") == 0)
            {
                version ();
                return 0;
            }
        else if (argc < 3)
            {
                usage ();
                return 1;
            }

        if (strcasecmp (argv[2], "--prefix") == 0)
            {
                PrefixDB *db;
                char *prefix;

                db = pdbInit (argv[1], NULL, NULL, NULL, NULL, NULL, NULL);
                if (!db)
                    {
                        if (argv[3])
                            printf ("%s\n", argv[3]);
                        else
                            {
                                fprintf (stderr, "pdbget: unable to read information for domain %s\n", argv[1]);
                                return 1;
                            }
                    }
                else
                    {
                        prefix = pdbGetPrefix (db);
                        if (!prefix)
                            {
                                if (argv[3])
                                        printf ("%s\n", argv[3]);
                                else
                                    {
                                        fprintf (stderr, "pdbget: unable to get prefix for %s\n", argv[1]);
                                        return 1;
                                    }
                            }
                        else
                                printf ("%s\n", prefix);
                    }
            }
        else
            {
                if (strcasecmp (argv[2], "--standard") == 0)
                    {
                        if (argc < 5)
                            {
                                fprintf (stderr, "pdbget: not enough arguments.\n");
                                return 1;
                            }
                        else
                            {
                                PrefixDB *db;

                                db = pdbInit (domain, argv[3], NULL, NULL, NULL, NULL, NULL);
                                if (strcasecmp (argv[4], "bin") == 0)
                                        printstd ("bin", pdbGetBin (db), argv[5]);
                                else if (strcasecmp (argv[4], "share") == 0)
                                        printstd ("share", pdbGetShare (db), argv[5]);
                                else if (strcasecmp (argv[4], "etc") == 0)
                                        printstd ("etc", pdbGetEtc (db), argv[5]);
                                else if (strcasecmp (argv[4], "lib") == 0)
                                        printstd ("lib", pdbGetLib (db), argv[5]);
                                else if (strcasecmp (argv[4], "libexec") == 0)
                                        printstd ("libexec", pdbGetLibexec (db), argv[5]);
                                else
                                    {
                                        fprintf (stderr, "%s is not a standard directory. Please use --custom instead.\n", argv[4]);
                                        return 1;
                                    }
                            }
                    }
                else if (strcasecmp (argv[2], "--custom") == 0)
                    {
                        if (argc < 4)
                            {
                                fprintf (stderr, "pdbget: not enough arguments.\n");
                                return 1;
                            }
                        else
                            {
                                PrefixDB *db;

                                db = pdbInit (domain, NULL, NULL, NULL, NULL, NULL, NULL);
                                if (!db)
                                    {
                                        if (argv[4])
                                            printf ("%s\n", argv[4]);
                                        else
                                            {
                                                fprintf (stderr, "pdbget: unable to read information for domain %s\n", argv[1]);
                                                return 1;
                                            }
                                    }
                                else
                                    {
                                        char *value;

                                        value = pdbGet (db, argv[3]);
                                        if (value)
                                                printf ("%s\n", value);
                                        else if (argv[4])
                                                printf ("%s\n", argv[4]);
                                        else
                                            {
                                                fprintf (stderr, "pdbget: information about '%s' not found\n", argv[3]);
                                                return 1;
                                            }
                                    }
                            }
                    }
                else
                    {
                        fprintf (stderr, "pdbget: invalid argument %s\n", argv[2]);
                        return 1;
                    }
            }


        return 0;
}
