/*
 *  libprefixdb
 *  Copyright (C) 2002-2003  Hongli Lai <h.lai@chello.nl>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <pwd.h>
#include <sys/types.h>


PrefixDB *
pdbInit (char *domain_name_version,
         char *prefix, char *bin, char *share,
         char *etc,    char *lib, char *libexec)
{
        PrefixDB *db;
        char *filename;
        struct passwd *pw;

        if (!domain_name_version)
                return NULL;

        filename = calloc (1, PATH_MAX + 1);
        if (!filename)
            {
                fprintf (stderr, "*ERROR*: prefixdb: unable to "
                         "allocate %d bytes of memory!\n", PATH_MAX + 1);
                return NULL;
            }


        /* First, look in ~/.packages */
        pw = getpwuid (getuid ());
        if (!pw || !pw->pw_dir)
                /* Huh??? Can't get user info?? Or no home dir?? */
                goto systemwide;
        snprintf (filename, PATH_MAX + 1, "%s/.packages/%s/prefixes",
            pw->pw_dir, domain_name_version);

        db = pdbInitFromFile (filename, prefix, bin, share, etc, lib, libexec);
        if (!db)
            {
                systemwide:
                /* Failed; look in PACKAGEDIR (usually /var/packages) */
                snprintf (filename, PATH_MAX + 1, PACKAGEDIR "/%s/prefixes",
                    domain_name_version);

                db = pdbInitFromFile (filename, prefix, bin, share, etc, lib, libexec);
                if (!db)
                    {
                        safe_free (filename);
                        return NULL;
                    }
                else
                        goto success;
            }
        else
            {
                success:
                /* Return db information from ~/.packages */
                db->domain_name_version  = safe_strdup (domain_name_version);
                safe_free (filename);
                return db;
            }
}


PrefixDB *
pdbInitFromFile (char *filename,
                 char *prefix, char *bin, char *share,
                 char *etc,    char *lib, char *libexec)
{
        PrefixDB *db;

        if (!filename)
                return NULL;

        db = calloc (sizeof (PrefixDB), 1);
        if (!db)
            {
                fprintf (stderr, "*ERROR*: pdbInit: unable to allocate %d "
                         "bytes of memory.\n", sizeof (PrefixDB));
                return NULL;
            }

        db->dbfile = safe_strdup (filename);
        if (!db->dbfile)
            {
                fprintf (stderr, "*ERROR*: prefixdb: unable to "
                         "perform strdup()!\n");
                safe_free (db);
                return NULL;
            }

        db->prefix  = safe_strdup (prefix);
        db->bin     = safe_strdup (bin);
        db->share   = safe_strdup (share);
        db->etc     = safe_strdup (etc);
        db->lib     = safe_strdup (lib);
        db->libexec = safe_strdup (libexec);

        /* Set default values */
        db->prefix_exists = 1;
        db->bin_exists = 1;
        db->share_exists = 1;
        db->etc_exists = 1;
        db->lib_exists = 1;
        db->libexec_exists = 1;

        /* Check for existance */
        db->prefix_exists = check_and_free (pdbGetPrefix (db));
        db->bin_exists = check_and_free (pdbGetBin (db));
        db->share_exists = check_and_free (pdbGetShare (db));
        db->etc_exists = check_and_free (pdbGetEtc (db));
        db->lib_exists = check_and_free (pdbGetLib (db));
        db->libexec_exists = check_and_free (pdbGetLibexec (db));

        return db;
}


void
pdbFree (PrefixDB *db)
{
        if (!db)
                return;

        safe_free (db->domain_name_version);

        safe_free (db->prefix);
        safe_free (db->bin);
        safe_free (db->share);
        safe_free (db->etc);
        safe_free (db->lib);
        safe_free (db->libexec);

        safe_free (db->dbfile);
        safe_free (db);
}
