/*
 *  libprefixdb
 *  Copyright (C) 2002-2003  Hongli Lai <h.lai@chello.nl>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


char *
pdbGetPrefix (PrefixDB *db)
{
        char *prefix = NULL;

        if (!db)
                return NULL;

        if (db->prefix_exists)
                prefix = pdbRead (db, "prefix");
        if (prefix)
                return prefix;
        else
                return safe_strdup (db->prefix);
}



/**************************************************
 * The following functions use this algorithm:
 * + Check for db->dir_exists.
 * |-+ If 1, check for a database entry key.
 *   |-  If non-NULL, return the key's value.
 *   |-+ If NULL, get the prefix using pdbGetPrefix() <this is step A>.
 *     |-  If prefix is not NULL, return "prefix/subdir".
 *     |-+ If prefix is NULL, check for hardcoded path in db->subdir.
 *       |- If db->bin is not NULL, return db->bin.
 *       |- Otherwise, return NULL.
 * |- If 0, goto step A.
 *
 **************************************************/

char *
pdbGetBin (PrefixDB *db)
{
        char *bin = NULL;

        if (!db)
                return NULL;

        if (db->bin_exists)
                bin = pdbRead (db, "bin");
        if (bin)
                return bin;
        else
            {
                char *prefix;

                prefix = pdbGetPrefix (db);
                if (prefix)
                    {
                        bin = calloc (1, PATH_MAX + 1);
                        snprintf (bin, PATH_MAX + 1, "%s/bin", prefix);
                        free (prefix);
                        return bin;
                    }
                else
                        return safe_strdup (db->bin);
            }
}


char *
pdbGetShare (PrefixDB *db)
{
        char *share = NULL;

        if (!db)
                return NULL;

        if (db->share_exists)
                share = pdbRead (db, "share");

        if (share)
                return share;
        else
            {
                char *prefix;

                prefix = pdbGetPrefix (db);
                if (prefix)
                    {
                        share = calloc (1, PATH_MAX + 1);
                        snprintf (share, PATH_MAX + 1, "%s/share", prefix);
                        free (prefix);
                        return share;
                    }
                else
                        return safe_strdup (db->share);
            }
}


char *
pdbGetEtc (PrefixDB *db)
{
        char *etc = NULL;

        if (!db)
                return NULL;

        if (db->etc_exists)
                etc = pdbRead (db, "etc");

        if (etc)
                return etc;
        else
            {
                char *prefix;

                prefix = pdbGetPrefix (db);
                if (prefix)
                    {
                        etc = calloc (1, PATH_MAX + 1);
                        snprintf (etc, PATH_MAX + 1, "%s/etc", prefix);
                        free (prefix);
                        return etc;
                    }
                else
                        return safe_strdup (db->etc);
            }
}


char *
pdbGetLib (PrefixDB *db)
{
        char *lib = NULL;

        if (!db)
                return NULL;

        if (db->lib_exists)
                lib = pdbRead (db, "lib");

        if (lib)
                return lib;
        else
            {
                char *prefix;

                prefix = pdbGetPrefix (db);
                if (prefix)
                    {
                        lib = calloc (1, PATH_MAX + 1);
                        snprintf (lib, PATH_MAX + 1, "%s/lib", prefix);
                        free (prefix);
                        return lib;
                    }
                else
                        return safe_strdup (db->lib);
            }
}


char *
pdbGetLibexec (PrefixDB *db)
{
        char *libexec = NULL;

        if (!db)
                return NULL;

        if (db->libexec_exists)
                libexec = pdbRead (db, "libexec");

        if (libexec)
                return libexec;
        else
            {
                char *prefix;

                prefix = pdbGetPrefix (db);
                if (prefix)
                    {
                        libexec = calloc (1, PATH_MAX + 1);
                        snprintf (libexec, PATH_MAX + 1, "%s/libexec", prefix);
                        free (prefix);
                        return libexec;
                    }
                else
                        return safe_strdup (db->libexec);
            }
}


char *
pdbGet (PrefixDB *db, char *name)
{
        if (!db || !name)
                return NULL;
        return pdbRead (db, name);
}


char *
pdbGetAndFbk (PrefixDB *db, char *name, char *fallback)
{
        char *value;

        if (!db || !name || !fallback)
                return NULL;
        value = pdbGet (db, name);
        return (value) ? value : strdup (fallback);
}
